﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/codeartifact/CodeArtifactRequest.h>
#include <aws/codeartifact/CodeArtifact_EXPORTS.h>
#include <aws/codeartifact/model/PackageFormat.h>
#include <aws/codeartifact/model/PackageVersionStatus.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>

#include <utility>

namespace Aws {
namespace Http {
class URI;
}  // namespace Http
namespace CodeArtifact {
namespace Model {

/**
 */
class DeletePackageVersionsRequest : public CodeArtifactRequest {
 public:
  AWS_CODEARTIFACT_API DeletePackageVersionsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DeletePackageVersions"; }

  AWS_CODEARTIFACT_API Aws::String SerializePayload() const override;

  AWS_CODEARTIFACT_API void AddQueryStringParameters(Aws::Http::URI& uri) const override;

  ///@{
  /**
   * <p> The name of the domain that contains the package to delete. </p>
   */
  inline const Aws::String& GetDomain() const { return m_domain; }
  inline bool DomainHasBeenSet() const { return m_domainHasBeenSet; }
  template <typename DomainT = Aws::String>
  void SetDomain(DomainT&& value) {
    m_domainHasBeenSet = true;
    m_domain = std::forward<DomainT>(value);
  }
  template <typename DomainT = Aws::String>
  DeletePackageVersionsRequest& WithDomain(DomainT&& value) {
    SetDomain(std::forward<DomainT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The 12-digit account number of the Amazon Web Services account that owns the
   * domain. It does not include dashes or spaces. </p>
   */
  inline const Aws::String& GetDomainOwner() const { return m_domainOwner; }
  inline bool DomainOwnerHasBeenSet() const { return m_domainOwnerHasBeenSet; }
  template <typename DomainOwnerT = Aws::String>
  void SetDomainOwner(DomainOwnerT&& value) {
    m_domainOwnerHasBeenSet = true;
    m_domainOwner = std::forward<DomainOwnerT>(value);
  }
  template <typename DomainOwnerT = Aws::String>
  DeletePackageVersionsRequest& WithDomainOwner(DomainOwnerT&& value) {
    SetDomainOwner(std::forward<DomainOwnerT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The name of the repository that contains the package versions to delete.
   * </p>
   */
  inline const Aws::String& GetRepository() const { return m_repository; }
  inline bool RepositoryHasBeenSet() const { return m_repositoryHasBeenSet; }
  template <typename RepositoryT = Aws::String>
  void SetRepository(RepositoryT&& value) {
    m_repositoryHasBeenSet = true;
    m_repository = std::forward<RepositoryT>(value);
  }
  template <typename RepositoryT = Aws::String>
  DeletePackageVersionsRequest& WithRepository(RepositoryT&& value) {
    SetRepository(std::forward<RepositoryT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The format of the package versions to delete. </p>
   */
  inline PackageFormat GetFormat() const { return m_format; }
  inline bool FormatHasBeenSet() const { return m_formatHasBeenSet; }
  inline void SetFormat(PackageFormat value) {
    m_formatHasBeenSet = true;
    m_format = value;
  }
  inline DeletePackageVersionsRequest& WithFormat(PackageFormat value) {
    SetFormat(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The namespace of the package versions to be deleted. The package component
   * that specifies its namespace depends on its type. For example:</p>  <p>The
   * namespace is required when deleting package versions of the following
   * formats:</p> <ul> <li> <p>Maven</p> </li> <li> <p>Swift</p> </li> <li>
   * <p>generic</p> </li> </ul>  <ul> <li> <p> The namespace of a Maven
   * package version is its <code>groupId</code>. </p> </li> <li> <p> The namespace
   * of an npm or Swift package version is its <code>scope</code>. </p> </li> <li>
   * <p>The namespace of a generic package is its <code>namespace</code>.</p> </li>
   * <li> <p> Python, NuGet, Ruby, and Cargo package versions do not contain a
   * corresponding component, package versions of those formats do not have a
   * namespace. </p> </li> </ul>
   */
  inline const Aws::String& GetNamespace() const { return m_namespace; }
  inline bool NamespaceHasBeenSet() const { return m_namespaceHasBeenSet; }
  template <typename NamespaceT = Aws::String>
  void SetNamespace(NamespaceT&& value) {
    m_namespaceHasBeenSet = true;
    m_namespace = std::forward<NamespaceT>(value);
  }
  template <typename NamespaceT = Aws::String>
  DeletePackageVersionsRequest& WithNamespace(NamespaceT&& value) {
    SetNamespace(std::forward<NamespaceT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The name of the package with the versions to delete. </p>
   */
  inline const Aws::String& GetPackage() const { return m_package; }
  inline bool PackageHasBeenSet() const { return m_packageHasBeenSet; }
  template <typename PackageT = Aws::String>
  void SetPackage(PackageT&& value) {
    m_packageHasBeenSet = true;
    m_package = std::forward<PackageT>(value);
  }
  template <typename PackageT = Aws::String>
  DeletePackageVersionsRequest& WithPackage(PackageT&& value) {
    SetPackage(std::forward<PackageT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> An array of strings that specify the versions of the package to delete. </p>
   */
  inline const Aws::Vector<Aws::String>& GetVersions() const { return m_versions; }
  inline bool VersionsHasBeenSet() const { return m_versionsHasBeenSet; }
  template <typename VersionsT = Aws::Vector<Aws::String>>
  void SetVersions(VersionsT&& value) {
    m_versionsHasBeenSet = true;
    m_versions = std::forward<VersionsT>(value);
  }
  template <typename VersionsT = Aws::Vector<Aws::String>>
  DeletePackageVersionsRequest& WithVersions(VersionsT&& value) {
    SetVersions(std::forward<VersionsT>(value));
    return *this;
  }
  template <typename VersionsT = Aws::String>
  DeletePackageVersionsRequest& AddVersions(VersionsT&& value) {
    m_versionsHasBeenSet = true;
    m_versions.emplace_back(std::forward<VersionsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The expected status of the package version to delete. </p>
   */
  inline PackageVersionStatus GetExpectedStatus() const { return m_expectedStatus; }
  inline bool ExpectedStatusHasBeenSet() const { return m_expectedStatusHasBeenSet; }
  inline void SetExpectedStatus(PackageVersionStatus value) {
    m_expectedStatusHasBeenSet = true;
    m_expectedStatus = value;
  }
  inline DeletePackageVersionsRequest& WithExpectedStatus(PackageVersionStatus value) {
    SetExpectedStatus(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_domain;
  bool m_domainHasBeenSet = false;

  Aws::String m_domainOwner;
  bool m_domainOwnerHasBeenSet = false;

  Aws::String m_repository;
  bool m_repositoryHasBeenSet = false;

  PackageFormat m_format{PackageFormat::NOT_SET};
  bool m_formatHasBeenSet = false;

  Aws::String m_namespace;
  bool m_namespaceHasBeenSet = false;

  Aws::String m_package;
  bool m_packageHasBeenSet = false;

  Aws::Vector<Aws::String> m_versions;
  bool m_versionsHasBeenSet = false;

  PackageVersionStatus m_expectedStatus{PackageVersionStatus::NOT_SET};
  bool m_expectedStatusHasBeenSet = false;
};

}  // namespace Model
}  // namespace CodeArtifact
}  // namespace Aws
